/**
 * CAS Quote Form Admin JavaScript - Repeater Version
 */

jQuery(document).ready(function($) {
    'use strict';
    
    let ports = [];
    let portCounter = 0;
    let pricing = [];
    let pricingCounter = 0;
    
    // Initialize admin functionality
    initAdminInterface();
    
    function initAdminInterface() {
        console.log('CAS Quote Form Admin initialized');
        
        // Load ports on page load
        loadPorts();
        
        // Add click handlers
        $('#add-port-row').on('click', function(e) {
            e.preventDefault();
            addPortRow();
        });
        
        $('#add-pricing-row').on('click', function(e) {
            e.preventDefault();
            addPricingRow();
        });
        
        $('#save-ports').on('click', function(e) {
            e.preventDefault();
            saveAllPorts();
        });
        
        $('#save-pricing').on('click', function(e) {
            e.preventDefault();
            saveAllPricing();
        });
        
        // Delegate event handlers for dynamic content
        $(document).on('click', '.cas-remove-port', function(e) {
            e.preventDefault();
            removePortRow($(this).closest('.cas-repeater-row'));
        });
        
        $(document).on('click', '.cas-remove-pricing', function(e) {
            e.preventDefault();
            removePricingRow($(this).closest('.cas-repeater-row'));
        });
        
        // Clear validation errors on input
        $(document).on('input', '.cas-port-name, .cas-port-state', function() {
            $(this).removeClass('cas-field-error');
            $(this).siblings('.cas-validation-error').remove();
        });
        
        $(document).on('input', '.cas-pricing-price', function() {
            $(this).removeClass('cas-field-error');
            $(this).siblings('.cas-validation-error').remove();
        });
        
        // Initialize drag and drop
        initDragAndDrop();
    }
    
    function loadPorts() {
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'cas_get_ports',
                nonce: cas_admin_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    ports = response.data || [];
                    renderPortRows();
                    // Load pricing after ports are loaded
                    loadPricing();
                } else {
                    showAdminNotice('Error loading ports: ' + response.data, 'error');
                }
            },
            error: function() {
                showAdminNotice('Error loading ports', 'error');
            }
        });
    }
    
    function renderPortRows() {
        const container = $('#cas-port-rows');
        container.empty();
        
        if (ports.length === 0) {
            container.html('<div class="cas-repeater-row cas-empty-row"><div class="cas-repeater-cell" style="flex: 1; justify-content: center;">No ports configured yet. Click "Add Port" to get started.</div></div>');
            return;
        }
        
        ports.forEach(function(port, index) {
            addPortRow(port, index);
        });
        
        // Reinitialize drag and drop after rendering
        initDragAndDrop();
    }
    
    function addPortRow(portData = null, index = null) {
        const container = $('#cas-port-rows');
        
        // Remove empty state if it exists
        container.find('.cas-empty-row').remove();
        
        const portId = portData ? portData.id : 'new_' + (++portCounter);
        const portName = portData ? portData.name : '';
        const portState = portData ? portData.state : '';
        
        const rowHtml = `
            <div class="cas-repeater-row cas-row-adding" data-port-id="${portId}">
                <div class="cas-repeater-cell cas-drag-cell">
                    <span class="cas-drag-handle">⋮⋮</span>
                </div>
                <div class="cas-repeater-cell">
                    <input type="text" class="cas-port-name" value="${portName}" placeholder="Enter port name">
                </div>
                <div class="cas-repeater-cell">
                    <input type="text" class="cas-port-state" value="${portState}" placeholder="Enter state">
                </div>
                <div class="cas-repeater-cell">
                    <div class="cas-action-buttons">
                        <button type="button" class="button button-small cas-remove-port cas-delete-button">Remove</button>
                    </div>
                </div>
            </div>
        `;
        
        if (index !== null) {
            container.append(rowHtml);
        } else {
            container.append(rowHtml);
        }
        
        // Trigger animation
        setTimeout(function() {
            container.find('.cas-row-adding').addClass('cas-row-added');
        }, 10);
    }
    
    function removePortRow($row) {
        $row.addClass('cas-row-removing');
        
        setTimeout(function() {
            $row.remove();
            
            // Show empty state if no rows left
            if ($('#cas-port-rows .cas-repeater-row').length === 0) {
                $('#cas-port-rows').html('<div class="cas-repeater-row cas-empty-row"><div class="cas-repeater-cell" style="flex: 1; justify-content: center;">No ports configured yet. Click "Add Port" to get started.</div></div>');
            }
        }, 200);
    }
    
    function saveAllPorts() {
        const $saveButton = $('#save-ports');
        const originalText = $saveButton.text();
        
        $saveButton.prop('disabled', true).text('Saving...');
        
        let portsData = [];
        let hasErrors = false;
        
        // Clear previous validation errors
        $('.cas-port-name, .cas-port-state').removeClass('cas-field-error');
        $('.cas-validation-error').remove();
        
        // Check if we have any actual port rows (not empty state)
        const portRows = $('#cas-port-rows .cas-repeater-row:not(.cas-empty-row)');
        
        if (portRows.length === 0) {
            // No ports to save, just save empty array
            portsData = [];
        } else {
            portRows.each(function() {
            const $row = $(this);
            const portId = $row.data('port-id');
            const portName = $row.find('.cas-port-name').val().trim();
            const portState = $row.find('.cas-port-state').val().trim();
            
            // Validate required fields
            let rowHasErrors = false;
            
            if (!portName) {
                $row.find('.cas-port-name').addClass('cas-field-error');
                showFieldError($row.find('.cas-port-name'), 'Port Name is required');
                rowHasErrors = true;
            }
            
            if (!portState) {
                $row.find('.cas-port-state').addClass('cas-field-error');
                showFieldError($row.find('.cas-port-state'), 'State is required');
                rowHasErrors = true;
            }
            
            if (rowHasErrors) {
                hasErrors = true;
                return; // Skip this row
            }
            
            portsData.push({
                port_id: portId,
                port_name: portName,
                port_state: portState
            });
            });
        }
        
        if (hasErrors) {
            $saveButton.prop('disabled', false).text(originalText);
            showAdminNotice('Please fill in all required fields (Port Name and State)', 'error');
            return;
        }
        
        console.log('Saving ports data:', portsData);
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'cas_save_ports',
                nonce: cas_admin_ajax.nonce,
                ports: portsData
            },
            success: function(response) {
                console.log('AJAX Success:', response);
                if (response.success) {
                    showAdminNotice(response.data.message, 'success');
                    ports = response.data.ports;
                    
                    // Update row IDs for newly created ports
                    $('#cas-port-rows .cas-repeater-row').each(function(index) {
                        if (ports[index]) {
                            $(this).data('port-id', ports[index].id);
                        }
                    });
                } else {
                    showAdminNotice('Error: ' + response.data, 'error');
                }
            },
            error: function(xhr, status, error) {
                console.log('AJAX Error:', xhr, status, error);
                showAdminNotice('Error saving ports', 'error');
            },
            complete: function() {
                console.log('AJAX Complete');
                $saveButton.prop('disabled', false).text(originalText);
            }
        });
    }
    
    // Pricing Functions
    function loadPricing() {
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'cas_get_pricing',
                nonce: cas_admin_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    pricing = response.data || [];
                    renderPricingRows();
                } else {
                    showAdminNotice('Error loading pricing: ' + response.data, 'error');
                }
            },
            error: function() {
                showAdminNotice('Error loading pricing', 'error');
            }
        });
    }
    
    function renderPricingRows() {
        const container = $('#cas-pricing-rows');
        container.empty();
        
        console.log('Rendering pricing rows:', pricing);
        
        if (pricing.length === 0) {
            container.html('<div class="cas-repeater-row cas-empty-row"><div class="cas-repeater-cell" style="flex: 1; justify-content: center;">No pricing rules configured yet. Click "Add Pricing Rule" to get started.</div></div>');
            return;
        }
        
        pricing.forEach(function(priceRule, index) {
            console.log('Adding pricing row:', priceRule);
            addPricingRow(priceRule, index);
        });
        
        // Reinitialize drag and drop after rendering
        initDragAndDrop();
    }
    
    function addPricingRow(priceData = null, index = null) {
        const container = $('#cas-pricing-rows');
        
        // Remove empty state if it exists
        container.find('.cas-empty-row').remove();
        
        const priceId = priceData ? priceData.id : 'new_' + (++pricingCounter);
        const fromPort = priceData ? priceData.from_port : '';
        const toPort = priceData ? priceData.to_port : '';
        const price = priceData ? priceData.price : '';
        
        console.log('addPricingRow data:', { priceId, fromPort, toPort, price });
        console.log('Available ports:', ports);
        
        // Generate port options
        const portOptions = ports.map(port => {
            const portValue = `${port.name}, ${port.state}`;
            const isSelected = fromPort === portValue ? 'selected' : '';
            return `<option value="${portValue}" ${isSelected}>${portValue}</option>`;
        }).join('');
        
        const toPortOptions = ports.map(port => {
            const portValue = `${port.name}, ${port.state}`;
            const isSelected = toPort === portValue ? 'selected' : '';
            return `<option value="${portValue}" ${isSelected}>${portValue}</option>`;
        }).join('');
        
        const rowHtml = `
            <div class="cas-repeater-row cas-row-adding" data-price-id="${priceId}">
                <div class="cas-repeater-cell">
                    <select class="cas-pricing-from-port">
                        <option value="">Select Port</option>
                        ${portOptions}
                    </select>
                </div>
                <div class="cas-repeater-cell">
                    <select class="cas-pricing-to-port">
                        <option value="">Select Port</option>
                        ${toPortOptions}
                    </select>
                </div>
                <div class="cas-repeater-cell">
                    <div class="cas-currency-input">
                        <div class="cas-currency-symbol">$</div>
                        <input type="number" class="cas-pricing-price" value="${price}" placeholder="0.00" step="0.01" min="0">
                    </div>
                </div>
                <div class="cas-repeater-cell">
                    <div class="cas-action-buttons">
                        <button type="button" class="button button-small cas-remove-pricing cas-delete-button">Remove</button>
                    </div>
                </div>
            </div>
        `;
        
        if (index !== null) {
            container.append(rowHtml);
        } else {
            container.append(rowHtml);
        }
        
        // Trigger animation
        setTimeout(function() {
            container.find('.cas-row-adding').addClass('cas-row-added');
        }, 10);
    }
    
    function removePricingRow($row) {
        $row.addClass('cas-row-removing');
        
        setTimeout(function() {
            $row.remove();
            
            // Show empty state if no rows left
            if ($('#cas-pricing-rows .cas-repeater-row').length === 0) {
                $('#cas-pricing-rows').html('<div class="cas-repeater-row cas-empty-row"><div class="cas-repeater-cell" style="flex: 1; justify-content: center;">No pricing rules configured yet. Click "Add Pricing Rule" to get started.</div></div>');
            }
        }, 200);
    }
    
    function saveAllPricing() {
        const $saveButton = $('#save-pricing');
        const originalText = $saveButton.text();
        
        $saveButton.prop('disabled', true).text('Saving...');
        
        let pricingData = [];
        let hasErrors = false;
        
        // Clear previous validation errors
        $('.cas-pricing-from-port, .cas-pricing-to-port, .cas-pricing-price').removeClass('cas-field-error');
        $('.cas-validation-error').remove();
        
        // Check if we have any actual pricing rows (not empty state)
        const pricingRows = $('#cas-pricing-rows .cas-repeater-row:not(.cas-empty-row)');
        
        if (pricingRows.length === 0) {
            // No pricing rules to save, just save empty array
            pricingData = [];
        } else {
            pricingRows.each(function() {
                const $row = $(this);
                const priceId = $row.data('price-id');
                const fromPort = $row.find('.cas-pricing-from-port').val();
                const toPort = $row.find('.cas-pricing-to-port').val();
                const price = parseFloat($row.find('.cas-pricing-price').val());
                
                // Validate required fields
                let rowHasErrors = false;
                
                if (!fromPort) {
                    $row.find('.cas-pricing-from-port').addClass('cas-field-error');
                    showFieldError($row.find('.cas-pricing-from-port'), 'From Port is required');
                    rowHasErrors = true;
                }
                
                if (!toPort) {
                    $row.find('.cas-pricing-to-port').addClass('cas-field-error');
                    showFieldError($row.find('.cas-pricing-to-port'), 'To Port is required');
                    rowHasErrors = true;
                }
                
                if (!price || price <= 0) {
                    $row.find('.cas-pricing-price').addClass('cas-field-error');
                    showFieldError($row.find('.cas-pricing-price'), 'Price must be greater than 0');
                    rowHasErrors = true;
                }
                
                if (rowHasErrors) {
                    hasErrors = true;
                    return; // Skip this row
                }
                
                pricingData.push({
                    price_id: priceId,
                    from_port: fromPort,
                    to_port: toPort,
                    price: price
                });
            });
        }
        
        if (hasErrors) {
            $saveButton.prop('disabled', false).text(originalText);
            showAdminNotice('Please fill in all required fields (From Port, To Port, and Price)', 'error');
            return;
        }
        
        console.log('Saving pricing data:', pricingData);
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'cas_save_pricing',
                nonce: cas_admin_ajax.nonce,
                pricing: pricingData
            },
            success: function(response) {
                console.log('AJAX Success:', response);
                if (response.success) {
                    showAdminNotice(response.data.message, 'success');
                    pricing = response.data.pricing;
                    
                    // Update row IDs for newly created pricing rules
                    $('#cas-pricing-rows .cas-repeater-row').each(function(index) {
                        if (pricing[index]) {
                            $(this).data('price-id', pricing[index].id);
                        }
                    });
                } else {
                    showAdminNotice('Error: ' + response.data, 'error');
                }
            },
            error: function(xhr, status, error) {
                console.log('AJAX Error:', xhr, status, error);
                showAdminNotice('Error saving pricing', 'error');
            },
            complete: function() {
                console.log('AJAX Complete');
                $saveButton.prop('disabled', false).text(originalText);
            }
        });
    }
    
    function initDragAndDrop() {
        // Make port rows sortable using jQuery UI Sortable
        $('#cas-port-rows').sortable({
            handle: '.cas-drag-handle',
            placeholder: 'cas-sortable-placeholder',
            helper: 'clone',
            opacity: 0.8,
            cursor: 'move',
            tolerance: 'pointer',
            start: function(event, ui) {
                ui.placeholder.height(ui.item.height());
                ui.item.addClass('cas-dragging');
            },
            stop: function(event, ui) {
                ui.item.removeClass('cas-dragging');
                // Optional: Add visual feedback that order changed
                $('#cas-port-rows .cas-repeater-row').addClass('cas-order-changed');
                setTimeout(function() {
                    $('#cas-port-rows .cas-repeater-row').removeClass('cas-order-changed');
                }, 1000);
            }
        });
        
    }
    
    function showFieldError($field, message) {
        // Remove existing error message
        $field.siblings('.cas-validation-error').remove();
        
        // Add error message
        const errorDiv = $('<div class="cas-validation-error">' + message + '</div>');
        $field.after(errorDiv);
    }
    
    function showAdminNotice(message, type = 'info') {
        const noticeClass = type === 'error' ? 'notice-error' : 
                           type === 'success' ? 'notice-success' : 
                           type === 'warning' ? 'notice-warning' : 'notice-info';
        
        const notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>');
        
        $('.wrap h1').after(notice);
        
        // Auto-dismiss after 5 seconds
        setTimeout(function() {
            notice.fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
    }
    
    // Export functions for global access
    window.CASAdmin = {
        showNotice: showAdminNotice,
        init: initAdminInterface,
        loadPorts: loadPorts,
        addPortRow: addPortRow,
        saveAllPorts: saveAllPorts,
        loadPricing: loadPricing,
        addPricingRow: addPricingRow,
        saveAllPricing: saveAllPricing
    };
});
