<?php
/**
 * WordPress Admin Interface for CAS Quote Form
 * 
 * @package CASQuoteForm
 * @version 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class CASQuoteFormAdmin {
    
    public function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('wp_ajax_cas_save_ports', [$this, 'ajax_save_ports']);
        add_action('wp_ajax_cas_get_ports', [$this, 'ajax_get_ports']);
        add_action('wp_ajax_cas_save_pricing', [$this, 'ajax_save_pricing']);
        add_action('wp_ajax_cas_get_pricing', [$this, 'ajax_get_pricing']);
    }
    
    /**
     * Add admin menu pages
     */
    public function add_admin_menu() {
        // Main menu page
        add_menu_page(
            'Quote Form',                    // Page title
            'Quote Form',                    // Menu title
            'manage_options',                // Capability
            'cas-quote-form',                // Menu slug
            [$this, 'quote_form_page'],      // Callback function
            'dashicons-car',        // Icon
            30                               // Position
        );
        
        // Submenu: Overview
        add_submenu_page(
            'cas-quote-form',                // Parent slug
            'Overview',                      // Page title
            'Overview',                      // Menu title
            'manage_options',                // Capability
            'cas-quote-form',                // Menu slug (same as parent)
            [$this, 'quote_form_page']       // Callback function
        );
        
        // Submenu: Ports & Pricing
        add_submenu_page(
            'cas-quote-form',                // Parent slug
            'Ports & Pricing',               // Page title
            'Ports & Pricing',               // Menu title
            'manage_options',                // Capability
            'cas-quote-form-ports',          // Menu slug
            [$this, 'ports_pricing_page']    // Callback function
        );

        // Submenu: Settings
        add_submenu_page(
            'cas-quote-form',
            'Settings',
            'Settings',
            'manage_options',
            'cas-quote-form-settings',
            [$this, 'settings_page']
        );
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our admin pages
        if (strpos($hook, 'cas-quote-form') === false) {
            return;
        }
        
        wp_enqueue_style(
            'cas-quote-form-admin',
            plugin_dir_url(__FILE__) . 'admin.css',
            [],
            '1.0.0'
        );
        
        wp_enqueue_script(
            'cas-quote-form-admin',
            plugin_dir_url(__FILE__) . 'admin.js',
            ['jquery', 'jquery-ui-sortable'],
            '1.0.0',
            true
        );
        
        // Localize script with AJAX data
        wp_localize_script('cas-quote-form-admin', 'cas_admin_ajax', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cas_admin_nonce')
        ]);
    }
    
    /**
     * Main Quote Form page
     */
    public function quote_form_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="cas-admin-dashboard">
                <div class="cas-admin-grid">
                    <!-- Quick Stats -->
                    <div class="cas-admin-card">
                        <h2>Quick Stats</h2>
                        <div class="cas-stats-grid">
                            <div class="cas-stat-item">
                                <span class="cas-stat-number">0</span>
                                <span class="cas-stat-label">Total Quotes</span>
                            </div>
                            <div class="cas-stat-item">
                                <span class="cas-stat-number">0</span>
                                <span class="cas-stat-label">This Month</span>
                            </div>
                            <div class="cas-stat-item">
                                <span class="cas-stat-number">0</span>
                                <span class="cas-stat-label">Pending</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Recent Activity -->
                    <div class="cas-admin-card">
                        <h2>Recent Activity</h2>
                        <p>No recent quotes found.</p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Settings page
     */
    public function settings_page() {
        if (!current_user_can('manage_options')) { return; }

        // Handle save
        if (!empty($_POST['cas_settings_submit']) && check_admin_referer('cas_settings_save', 'cas_settings_nonce')) {
            $settings = [
                'stripe_secret_key'      => sanitize_text_field($_POST['stripe_secret_key'] ?? ''),
                'stripe_publishable_key' => sanitize_text_field($_POST['stripe_publishable_key'] ?? ''),
                'stripe_webhook_secret'  => sanitize_text_field($_POST['stripe_webhook_secret'] ?? ''),
                'stripe_currency'        => sanitize_text_field($_POST['stripe_currency'] ?? 'usd'),
                'stripe_success_url'     => esc_url_raw($_POST['stripe_success_url'] ?? ''),
                'stripe_cancel_url'      => esc_url_raw($_POST['stripe_cancel_url'] ?? ''),
                'admin_email_notification' => sanitize_email($_POST['admin_email_notification'] ?? ''),
            ];
            update_option('cas_settings', $settings, false);
            echo '<div class="updated"><p>Settings saved.</p></div>';
        }

        $settings = get_option('cas_settings', [
            'stripe_secret_key' => '',
            'stripe_publishable_key' => '',
            'stripe_webhook_secret' => '',
            'stripe_currency' => 'usd',
            'stripe_success_url' => '',
            'stripe_cancel_url' => '',
            'admin_email_notification' => '',
        ]);
        ?>
        <div class="wrap">
            <h1>Quote Form Settings</h1>
            <form method="post" action="">
                <?php wp_nonce_field('cas_settings_save', 'cas_settings_nonce'); ?>
                <div class="cas-admin-dashboard">
                    <div class="cas-admin-card" style="max-width: 900px;">
                        <h2>Stripe Configuration</h2>
                        <table class="form-table" role="presentation">
                            <tbody>
                                <tr>
                                    <th scope="row"><label for="stripe_secret_key">Secret Key</label></th>
                                    <td><input type="password" class="regular-text" id="stripe_secret_key" name="stripe_secret_key" value="<?php echo esc_attr($settings['stripe_secret_key']); ?>" autocomplete="off"></td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="stripe_publishable_key">Publishable Key</label></th>
                                    <td><input type="password" class="regular-text" id="stripe_publishable_key" name="stripe_publishable_key" value="<?php echo esc_attr($settings['stripe_publishable_key']); ?>" autocomplete="off"></td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="stripe_webhook_secret">Webhook Signing Secret</label></th>
                                    <td><input type="password" class="regular-text" id="stripe_webhook_secret" name="stripe_webhook_secret" value="<?php echo esc_attr($settings['stripe_webhook_secret']); ?>" autocomplete="off">
                                        <p class="description">Create a webhook in Stripe to <code><?php echo esc_html(rest_url('cas/v1/stripe-webhook')); ?></code> and paste the signing secret here.</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="stripe_currency">Currency</label></th>
                                    <td><input type="text" class="regular-text" id="stripe_currency" name="stripe_currency" value="<?php echo esc_attr($settings['stripe_currency']); ?>" placeholder="usd"></td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="stripe_success_url">Success URL</label></th>
                                    <td><input type="url" class="regular-text" id="stripe_success_url" name="stripe_success_url" value="<?php echo esc_attr($settings['stripe_success_url']); ?>" placeholder="https://example.com/booking/success"></td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="stripe_cancel_url">Cancel URL</label></th>
                                    <td><input type="url" class="regular-text" id="stripe_cancel_url" name="stripe_cancel_url" value="<?php echo esc_attr($settings['stripe_cancel_url']); ?>" placeholder="https://example.com/booking/cancel"></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>

                    <div class="cas-admin-card" style="max-width: 900px; margin-top: 20px;">
                        <h2>Email Configuration</h2>
                        <table class="form-table" role="presentation">
                            <tbody>
                                <tr>
                                    <th scope="row"><label for="admin_email_notification">Admin Email Notification</label></th>
                                    <td>
                                        <input type="email" class="regular-text" id="admin_email_notification" name="admin_email_notification" value="<?php echo esc_attr($settings['admin_email_notification'] ?? ''); ?>" placeholder="admin@example.com">
                                        <p class="description">Email address to receive notifications for new quotes and bookings.</p>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
                <p style="margin-top: 20px;"><button type="submit" name="cas_settings_submit" value="1" class="button button-primary">Save Settings</button></p>
            </form>
        </div>
        <?php
    }
    
    /**
     * Ports & Pricing page
     */
    public function ports_pricing_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="cas-admin-dashboard">
                <div class="cas-admin-grid">
                    <!-- Port Management -->
                    <div class="cas-admin-card">
                        <h2>Port Management</h2>
                        <p>Manage shipping ports and their details.</p>
                        <!-- Port Repeater -->
                        <div class="cas-port-repeater">
                            <div class="cas-repeater-header">
                                <h3>Ports</h3>
                            </div>
                            
                            <div class="cas-repeater-content">
                                <div class="cas-repeater-table">
                                    <div class="cas-repeater-row cas-repeater-header-row">
                                        <div class="cas-repeater-cell cas-drag-cell"></div>
                                        <div class="cas-repeater-cell">Port Name <span class="cas-required">*</span></div>
                                        <div class="cas-repeater-cell">State <span class="cas-required">*</span></div>
                                        <div class="cas-repeater-cell">Actions</div>
                                    </div>
                                    <div id="cas-port-rows" class="cas-repeater-rows">
                                        <!-- Port rows will be added here -->
                                    </div>
                                </div>
                            </div>
                            
                            <div class="cas-repeater-footer">
                                <button type="button" class="button button-primary" id="add-port-row">Add Port</button>
                                <button type="button" class="button button-primary" id="save-ports">Save All Ports</button>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Pricing Matrix -->
                    <div class="cas-admin-card">
                        <h2>Pricing Matrix</h2>
                        <p>Configure port-to-port shipping rates.</p>
                        
                        <!-- Pricing Repeater -->
                        <div class="cas-pricing-repeater">
                            <div class="cas-repeater-header">
                                <h3>Pricing Rules</h3>
                            </div>
                            
                            <div class="cas-repeater-content">
                                <div class="cas-repeater-table">
                                    <div class="cas-repeater-row cas-repeater-header-row">
                                        <div class="cas-repeater-cell">From Port <span class="cas-required">*</span></div>
                                        <div class="cas-repeater-cell">To Port <span class="cas-required">*</span></div>
                                        <div class="cas-repeater-cell">Price <span class="cas-required">*</span></div>
                                        <div class="cas-repeater-cell">Actions</div>
                                    </div>
                                    <div id="cas-pricing-rows" class="cas-repeater-rows">
                                        <!-- Pricing rows will be added here -->
                                    </div>
                                </div>
                            </div>
                            
                            <div class="cas-repeater-footer">
                                <button type="button" class="button button-primary" id="add-pricing-row">Add Pricing Rule</button>
                                <button type="button" class="button button-primary" id="save-pricing">Save All Pricing Rules</button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * AJAX: Save all ports
     */
    public function ajax_save_ports() {
        check_ajax_referer('cas_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $ports_data = isset($_POST['ports']) ? $_POST['ports'] : [];
        $ports = [];
        
        // Handle empty ports array
        if (empty($ports_data)) {
            $ports = [];
        } else {
            foreach ($ports_data as $port_data) {
            $port_name = sanitize_text_field($port_data['port_name']);
            $port_state = sanitize_text_field($port_data['port_state']);
            
            if (!empty($port_name) && !empty($port_state)) {
                $ports[] = [
                    'id' => isset($port_data['port_id']) ? sanitize_text_field($port_data['port_id']) : uniqid(),
                    'name' => $port_name,
                    'state' => $port_state,
                    'updated_at' => current_time('mysql')
                ];
            }
            }
        }
        
        update_option('cas_ports', $ports);
        
        wp_send_json_success([
            'message' => 'Ports saved successfully',
            'ports' => $ports
        ]);
    }
    
    /**
     * AJAX: Get all ports
     */
    public function ajax_get_ports() {
        check_ajax_referer('cas_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $ports = get_option('cas_ports', []);
        wp_send_json_success($ports);
    }
    
    /**
     * AJAX: Save all pricing rules
     */
    public function ajax_save_pricing() {
        check_ajax_referer('cas_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $pricing_data = isset($_POST['pricing']) ? $_POST['pricing'] : [];
        $pricing = [];
        
        // Handle empty pricing array
        if (empty($pricing_data)) {
            $pricing = [];
        } else {
            foreach ($pricing_data as $price_data) {
                $from_port = sanitize_text_field($price_data['from_port']);
                $to_port = sanitize_text_field($price_data['to_port']);
                $price = floatval($price_data['price']);
                
                if (!empty($from_port) && !empty($to_port) && $price > 0) {
                    $pricing[] = [
                        'id' => isset($price_data['price_id']) ? sanitize_text_field($price_data['price_id']) : uniqid(),
                        'from_port' => $from_port,
                        'to_port' => $to_port,
                        'price' => $price,
                        'updated_at' => current_time('mysql')
                    ];
                }
            }
        }
        
        update_option('cas_pricing', $pricing);
        
        wp_send_json_success([
            'message' => 'Pricing rules saved successfully',
            'pricing' => $pricing
        ]);
    }
    
    /**
     * AJAX: Get all pricing rules
     */
    public function ajax_get_pricing() {
        check_ajax_referer('cas_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $pricing = get_option('cas_pricing', []);
        wp_send_json_success($pricing);
    }
}

// Initialize the admin class
new CASQuoteFormAdmin();
