<?php
/**
 * Plugin Name: Coastal Auto Shipping Quote Form
 * Description: A plugin to handle the quote form for Coastal Auto Shipping.
 * Version: 1.0.0
 * Author: On The Map
 * Author URI: https://onthemap.com
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class CASQuoteForm {
    private $api_token = 'zghv93tb4gfdnfcz0ecz98baepna1917z1lfk6lr';
    private $api_base_url = 'https://rateengine.ship.cars/v2';
    private $google_places_api_key = 'AIzaSyBDZtvK3P2IplPFEYEJ523pycjWhyFNWe8';

    public function __construct() {
        // Use REST API instead of AJAX
        add_action('rest_api_init', [$this, 'register_rest_routes']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_google_maps']);
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('cas/v1', '/vehicle-years', [
            'methods' => 'GET',
            'callback' => [$this, 'get_vehicle_years'],
            'permission_callback' => '__return_true',
        ]);

        register_rest_route('cas/v1', '/vehicle-makes', [
            'methods' => 'GET',
            'callback' => [$this, 'get_vehicle_makes'],
            'permission_callback' => '__return_true',
            'args' => [
                'year' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
            ],
        ]);

        register_rest_route('cas/v1', '/vehicle-models', [
            'methods' => 'GET',
            'callback' => [$this, 'get_vehicle_models'],
            'permission_callback' => '__return_true',
            'args' => [
                'year' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
                'make' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
            ],
        ]);

        // Google Places API endpoints
        register_rest_route('cas/v1', '/places-autocomplete', [
            'methods' => 'GET',
            'callback' => [$this, 'get_places_autocomplete'],
            'permission_callback' => '__return_true',
            'args' => [
                'input' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
                'types' => [
                    'required' => false,
                    'type' => 'string',
                    'default' => 'address',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
            ],
        ]);

        register_rest_route('cas/v1', '/places-details', [
            'methods' => 'GET',
            'callback' => [$this, 'get_places_details'],
            'permission_callback' => '__return_true',
            'args' => [
                'place_id' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
            ],
        ]);

        register_rest_route('cas/v1', '/ports', [
            'methods' => 'GET',
            'callback' => [$this, 'get_ports'],
            'permission_callback' => '__return_true',
        ]);

        register_rest_route('cas/v1', '/pricing', [
            'methods' => 'GET',
            'callback' => [$this, 'get_pricing'],
            'permission_callback' => '__return_true',
        ]);

        register_rest_route('cas/v1', '/submit-quote', [
            'methods' => 'POST',
            'callback' => [$this, 'submit_quote_request'],
            'permission_callback' => '__return_true',
        ]);
    }

    /**
     * Enqueue Google Maps JavaScript API
     */
    public function enqueue_google_maps() {
        // Only enqueue on shipping quote page
        if (is_page('shipping-quote') || is_page_template('page-shipping-quote.php')) {
            wp_enqueue_script(
                'google-maps-places',
                'https://maps.googleapis.com/maps/api/js?key=' . $this->google_places_api_key . '&libraries=places&callback=initGoogleMaps',
                [],
                null,
                true
            );
            
            // Add inline script to initialize Google Maps
            wp_add_inline_script('google-maps-places', '
                window.initGoogleMaps = function() {
                    console.log("Google Maps Places API loaded");
                };
            ');
        }
    }

    /**
     * Test endpoint to verify plugin is working
     */
    public function test_plugin() {
        wp_send_json_success([
            'message' => 'CAS Quote Form plugin is working!',
            'timestamp' => current_time('mysql')
        ]);
    }

    /**
     * Get vehicle years from Ship.Cars API
     */
    public function get_vehicle_years($request) {
        // Add debugging
        error_log('CAS Quote Form: get_vehicle_years called');
        
        $url = $this->api_base_url . '/vehicles/years/?token=' . $this->api_token;
        error_log('CAS Quote Form: API URL: ' . $url);
        
        $response = wp_remote_get($url, [
            'headers' => [
                'Accept' => 'application/json',
            ],
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            error_log('CAS Quote Form: WP Error: ' . $response->get_error_message());
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Failed to fetch vehicle years: ' . $response->get_error_message()
            ], 500);
        }

        $response_code = wp_remote_retrieve_response_code($response);
        error_log('CAS Quote Form: Response code: ' . $response_code);
        
        $body = wp_remote_retrieve_body($response);
        error_log('CAS Quote Form: Response body: ' . substr($body, 0, 500));
        
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('CAS Quote Form: JSON Error: ' . json_last_error_msg());
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Invalid JSON response from API: ' . json_last_error_msg()
            ], 500);
        }

        // Ship.Cars API returns array of objects like [{"year": 2026}, {"year": 2025}]
        // Extract just the year values
        $years = array_map(function($item) {
            return $item['year'];
        }, $data);
        
        error_log('CAS Quote Form: Success, years count: ' . count($years));
        return new WP_REST_Response([
            'success' => true,
            'data' => [
                'years' => $years
            ]
        ], 200);
    }

    /**
     * Get vehicle makes from Ship.Cars API
     */
    public function get_vehicle_makes($request) {
        $year = $request->get_param('year');
        
        if (empty($year)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Year parameter is required'
            ], 400);
        }

        $url = $this->api_base_url . '/vehicles/makes/?year=' . urlencode($year) . '&token=' . $this->api_token;
        
        $response = wp_remote_get($url, [
            'headers' => [
                'Accept' => 'application/json',
            ],
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Failed to fetch vehicle makes: ' . $response->get_error_message()
            ], 500);
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Invalid JSON response from API'
            ], 500);
        }

        // Ship.Cars API returns array of objects like [{"make":"Acura"},{"make":"BMW"}]
        // Extract just the make values
        $makes = array_map(function($item) {
            return $item['make'];
        }, $data);
        
        return new WP_REST_Response([
            'success' => true,
            'data' => [
                'makes' => $makes
            ]
        ], 200);
    }

    /**
     * Get vehicle models from Ship.Cars API
     */
    public function get_vehicle_models($request) {
        $year = $request->get_param('year');
        $make = $request->get_param('make');
        
        if (empty($year) || empty($make)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Year and make parameters are required'
            ], 400);
        }

        $url = $this->api_base_url . '/vehicles/models/?year=' . urlencode($year) . '&make=' . urlencode($make) . '&token=' . $this->api_token;
        
        $response = wp_remote_get($url, [
            'headers' => [
                'Accept' => 'application/json',
            ],
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Failed to fetch vehicle models: ' . $response->get_error_message()
            ], 500);
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Invalid JSON response from API'
            ], 500);
        }

        // Ship.Cars API returns array of objects with detailed vehicle info
        // Extract just the model names and remove duplicates
        $models = array_unique(array_map(function($item) {
            return $item['model'];
        }, $data));
        
        // Sort models alphabetically
        sort($models);
        
        return new WP_REST_Response([
            'success' => true,
            'data' => [
                'models' => array_values($models) // Re-index array
            ]
        ], 200);
    }

    /**
     * Get Google Places Autocomplete suggestions
     */
    public function get_places_autocomplete($request) {
        $input = $request->get_param('input');
        $types = $request->get_param('types');

        if (empty($input)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Input parameter is required'
            ], 400);
        }

        $url = 'https://maps.googleapis.com/maps/api/place/autocomplete/json';
        $params = [
            'input' => $input,
            'components' => 'country:us', // Restrict to US only
            'key' => $this->google_places_api_key,
        ];

        $response = wp_remote_get($url . '?' . http_build_query($params), [
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Failed to fetch places: ' . $response->get_error_message()
            ], 500);
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Invalid JSON response from Google Places API'
            ], 500);
        }

        return new WP_REST_Response([
            'success' => true,
            'data' => $data
        ], 200);
    }

    /**
     * Get Google Places Details for formatted address
     */
    public function get_places_details($request) {
        $place_id = $request->get_param('place_id');

        if (empty($place_id)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Place ID parameter is required'
            ], 400);
        }

        $url = 'https://maps.googleapis.com/maps/api/place/details/json';
        $params = [
            'place_id' => $place_id,
            'fields' => 'formatted_address,address_components,geometry',
            'key' => $this->google_places_api_key,
        ];

        $response = wp_remote_get($url . '?' . http_build_query($params), [
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Failed to fetch place details: ' . $response->get_error_message()
            ], 500);
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Invalid JSON response from Google Places API'
            ], 500);
        }

        return new WP_REST_Response([
            'success' => true,
            'data' => $data
        ], 200);
    }

    /**
     * Get all ports from WordPress options
     */
    public function get_ports($request) {
        $ports = get_option('cas_ports', []);
        
        return new WP_REST_Response([
            'success' => true,
            'data' => $ports
        ], 200);
    }

    /**
     * Get all pricing rules from WordPress options
     */
    public function get_pricing($request) {
        $pricing = get_option('cas_pricing', []);
        
        return new WP_REST_Response([
            'success' => true,
            'data' => $pricing
        ], 200);
    }

    /**
     * Submit quote request (when no price is available)
     */
    public function submit_quote_request($request) {
        // Sanitize and get form data
        $origin_mode = sanitize_text_field($request->get_param('origin_mode') ?? 'city');
        $dest_mode = sanitize_text_field($request->get_param('dest_mode') ?? 'city');
        $from_city_zip = sanitize_text_field($request->get_param('from_city_zip') ?? '');
        $to_city_zip = sanitize_text_field($request->get_param('to_city_zip') ?? '');
        $from_port = sanitize_text_field($request->get_param('from_port') ?? '');
        $to_port = sanitize_text_field($request->get_param('to_port') ?? '');
        $first_name = sanitize_text_field($request->get_param('first_name') ?? '');
        $last_name = sanitize_text_field($request->get_param('last_name') ?? '');
        $email = sanitize_email($request->get_param('email') ?? '');
        $phone = sanitize_text_field($request->get_param('phone') ?? '');
        $pickup_date = sanitize_text_field($request->get_param('pickup_date') ?? '');
        
        // Get vehicles
        $vehicles_json = $request->get_param('vehicles');
        $vehicles = [];
        if (!empty($vehicles_json)) {
            $vehicles = json_decode(stripslashes($vehicles_json), true) ?? [];
            // Sanitize vehicle data
            foreach ($vehicles as &$vehicle) {
                $vehicle = [
                    'year' => sanitize_text_field($vehicle['year'] ?? ''),
                    'make' => sanitize_text_field($vehicle['make'] ?? ''),
                    'model' => sanitize_text_field($vehicle['model'] ?? ''),
                    'operable' => !empty($vehicle['operable']),
                ];
            }
        }

        // Determine from/to locations
        $from_location = ($origin_mode === 'port') ? $from_port : $from_city_zip;
        $to_location = ($dest_mode === 'port') ? $to_port : $to_city_zip;

        // Send admin email notification
        CASQuoteEmailHelper::send_admin_notification('quote', [
            'from_location' => $from_location,
            'to_location' => $to_location,
            'vehicles' => $vehicles,
            'first_name' => $first_name,
            'last_name' => $last_name,
            'email' => $email,
            'phone' => $phone,
            'pickup_date' => $pickup_date,
        ]);

        return new WP_REST_Response([
            'success' => true,
            'message' => 'Quote request submitted successfully'
        ], 200);
    }
}

// Initialize the plugin
error_log('CAS Quote Form: Plugin file loaded');
new CASQuoteForm();

// Include admin interface
if (is_admin()) {
    require_once plugin_dir_path(__FILE__) . 'admin.php';
}

// Include email helper
require_once plugin_dir_path(__FILE__) . 'email-helper.php';

// Include payment endpoints
require_once plugin_dir_path(__FILE__) . 'payment.php';