<?php
/**
 * CAS Quote Form - Email Helper
 * 
 * Handles sending admin email notifications for bookings and quote requests.
 */

if (!defined('ABSPATH')) { exit; }

class CASQuoteEmailHelper {
    
    /**
     * Get admin email from settings
     */
    private static function get_admin_email() {
        $settings = get_option('cas_settings', []);
        return !empty($settings['admin_email_notification']) ? sanitize_email($settings['admin_email_notification']) : get_option('admin_email');
    }
    
    /**
     * Build email HTML content
     */
    private static function build_email_html($type, $data) {
        $is_booking = ($type === 'booking');
        $vehicles = $data['vehicles'] ?? [];
        $vehicle_count = count($vehicles);
        
        ob_start();
        ?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo esc_html($is_booking ? 'New Booking Received' : 'New Quote Request'); ?></title>
</head>
<body style="margin: 0; padding: 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif; background-color: #f5f5f5;">
    <table role="presentation" style="width: 100%; border-collapse: collapse; background-color: #f5f5f5; padding: 20px;">
        <tr>
            <td align="center">
                <table role="presentation" style="max-width: 600px; width: 100%; border-collapse: collapse; background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <!-- Header -->
                    <tr>
                        <td style="background-color: #ea580c; padding: 30px 20px; text-align: center;">
                            <h1 style="margin: 0; color: #ffffff; font-size: 24px; font-weight: bold;"><?php echo esc_html($is_booking ? 'New Booking Received' : 'New Quote Request'); ?></h1>
                        </td>
                    </tr>
                    
                    <!-- Content -->
                    <tr>
                        <td style="padding: 30px 20px;">
                            <p style="margin: 0 0 20px 0; color: #374151; font-size: 16px; line-height: 1.6;">
                                <?php echo esc_html($is_booking ? 'A new booking has been received with successful payment.' : 'A new quote request has been submitted.'); ?>
                            </p>

                            <!-- Shipping Route Details -->
                            <div style="background-color: #f9fafb; border-left: 4px solid #ea580c; padding: 20px; margin-bottom: 20px; border-radius: 4px;">
                                <h2 style="margin: 0 0 15px 0; color: #111827; font-size: 18px; font-weight: bold;">Shipping Route Details</h2>
                                <table role="presentation" style="width: 100%; border-collapse: collapse;">
                                    <tr>
                                        <td style="padding: 8px 0; color: #6b7280; font-size: 14px; width: 40%;">Transport From:</td>
                                        <td style="padding: 8px 0; color: #111827; font-size: 14px; font-weight: 600;"><?php echo esc_html($data['from_location'] ?? '—'); ?></td>
                                    </tr>
                                    <tr>
                                        <td style="padding: 8px 0; color: #6b7280; font-size: 14px;">Transport To:</td>
                                        <td style="padding: 8px 0; color: #111827; font-size: 14px; font-weight: 600;"><?php echo esc_html($data['to_location'] ?? '—'); ?></td>
                                    </tr>
                                </table>
                            </div>

                            <!-- Vehicle Details -->
                            <div style="background-color: #f9fafb; border-left: 4px solid #ea580c; padding: 20px; margin-bottom: 20px; border-radius: 4px;">
                                <h2 style="margin: 0 0 15px 0; color: #111827; font-size: 18px; font-weight: bold;">
                                    Vehicle Details 
                                    <span style="font-size: 14px; font-weight: normal; color: #6b7280;">(<?php echo esc_html($vehicle_count); ?> <?php echo esc_html($vehicle_count === 1 ? 'vehicle' : 'vehicles'); ?>)</span>
                                </h2>
                                <?php if (!empty($vehicles)): ?>
                                    <?php foreach ($vehicles as $index => $vehicle): ?>
                                        <div style="margin-bottom: <?php echo ($index === count($vehicles) - 1) ? '0' : '15px'; ?>; padding-bottom: <?php echo ($index === count($vehicles) - 1) ? '0' : '15px'; ?>; border-bottom: <?php echo ($index === count($vehicles) - 1) ? 'none' : '1px solid #e5e7eb'; ?>;">
                                            <h3 style="margin: 0 0 10px 0; color: #374151; font-size: 16px; font-weight: 600;">Vehicle <?php echo esc_html($index + 1); ?></h3>
                                            <table role="presentation" style="width: 100%; border-collapse: collapse;">
                                                <tr>
                                                    <td style="padding: 6px 0; color: #6b7280; font-size: 14px; width: 40%;">Year:</td>
                                                    <td style="padding: 6px 0; color: #111827; font-size: 14px;"><?php echo esc_html($vehicle['year'] ?? '—'); ?></td>
                                                </tr>
                                                <tr>
                                                    <td style="padding: 6px 0; color: #6b7280; font-size: 14px;">Make:</td>
                                                    <td style="padding: 6px 0; color: #111827; font-size: 14px;"><?php echo esc_html($vehicle['make'] ?? '—'); ?></td>
                                                </tr>
                                                <tr>
                                                    <td style="padding: 6px 0; color: #6b7280; font-size: 14px;">Model:</td>
                                                    <td style="padding: 6px 0; color: #111827; font-size: 14px;"><?php echo esc_html($vehicle['model'] ?? '—'); ?></td>
                                                </tr>
                                                <tr>
                                                    <td style="padding: 6px 0; color: #6b7280; font-size: 14px;">Operable:</td>
                                                    <td style="padding: 6px 0; color: #111827; font-size: 14px;"><?php echo (($vehicle['operable'] ?? false) ? 'Yes' : 'No'); ?></td>
                                                </tr>
                                            </table>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <p style="margin: 0; color: #6b7280; font-size: 14px;">No vehicle details provided.</p>
                                <?php endif; ?>
                            </div>

                            <!-- Contact Information -->
                            <div style="background-color: #f9fafb; border-left: 4px solid #ea580c; padding: 20px; margin-bottom: 20px; border-radius: 4px;">
                                <h2 style="margin: 0 0 15px 0; color: #111827; font-size: 18px; font-weight: bold;">Contact Information</h2>
                                <table role="presentation" style="width: 100%; border-collapse: collapse;">
                                    <tr>
                                        <td style="padding: 8px 0; color: #6b7280; font-size: 14px; width: 40%;">Name:</td>
                                        <td style="padding: 8px 0; color: #111827; font-size: 14px; font-weight: 600;"><?php echo esc_html(trim(($data['first_name'] ?? '') . ' ' . ($data['last_name'] ?? ''))); ?></td>
                                    </tr>
                                    <tr>
                                        <td style="padding: 8px 0; color: #6b7280; font-size: 14px;">Email:</td>
                                        <td style="padding: 8px 0; color: #111827; font-size: 14px;">
                                            <a href="mailto:<?php echo esc_attr($data['email'] ?? ''); ?>" style="color: #ea580c; text-decoration: none;"><?php echo esc_html($data['email'] ?? '—'); ?></a>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td style="padding: 8px 0; color: #6b7280; font-size: 14px;">Phone:</td>
                                        <td style="padding: 8px 0; color: #111827; font-size: 14px;">
                                            <a href="tel:<?php echo esc_attr($data['phone'] ?? ''); ?>" style="color: #111827; text-decoration: none;"><?php echo esc_html($data['phone'] ?? '—'); ?></a>
                                        </td>
                                    </tr>
                                    <?php if (!empty($data['pickup_date'])): ?>
                                    <tr>
                                        <td style="padding: 8px 0; color: #6b7280; font-size: 14px;">Estimated Pickup Date:</td>
                                        <td style="padding: 8px 0; color: #111827; font-size: 14px;"><?php echo esc_html($data['pickup_date']); ?></td>
                                    </tr>
                                    <?php endif; ?>
                                </table>
                            </div>

                            <?php if ($is_booking): ?>
                            <!-- Payment Details -->
                            <div style="background-color: #f0fdf4; border-left: 4px solid #22c55e; padding: 20px; margin-bottom: 20px; border-radius: 4px;">
                                <h2 style="margin: 0 0 15px 0; color: #111827; font-size: 18px; font-weight: bold;">Payment Details</h2>
                                <table role="presentation" style="width: 100%; border-collapse: collapse;">
                                    <tr>
                                        <td style="padding: 8px 0; color: #6b7280; font-size: 14px; width: 40%;">Amount:</td>
                                        <td style="padding: 8px 0; color: #111827; font-size: 18px; font-weight: bold;">$<?php echo number_format(($data['amount_cents'] ?? 0) / 100, 2); ?> <?php echo strtoupper($data['currency'] ?? 'USD'); ?></td>
                                    </tr>
                                    <tr>
                                        <td style="padding: 8px 0; color: #6b7280; font-size: 14px;">Status:</td>
                                        <td style="padding: 8px 0; color: #22c55e; font-size: 14px; font-weight: 600;">Paid</td>
                                    </tr>
                                    <?php if (!empty($data['booking_id'])): ?>
                                    <tr>
                                        <td style="padding: 8px 0; color: #6b7280; font-size: 14px;">Booking ID:</td>
                                        <td style="padding: 8px 0; color: #111827; font-size: 14px; font-family: monospace;"><?php echo esc_html($data['booking_id']); ?></td>
                                    </tr>
                                    <?php endif; ?>
                                </table>
                            </div>
                            <?php endif; ?>

                            <!-- Footer -->
                            <p style="margin: 30px 0 0 0; padding-top: 20px; border-top: 1px solid #e5e7eb; color: #6b7280; font-size: 14px; text-align: center;">
                                This is an automated notification from the Coastal Auto Shipping quote form system.
                            </p>
                        </td>
                    </tr>
                </table>
            </td>
        </tr>
    </table>
</body>
</html>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Send admin email notification
     */
    public static function send_admin_notification($type, $data) {
        error_log('CAS Quote Form: send_admin_notification called - type: ' . $type);
        
        $admin_email = self::get_admin_email();
        
        if (empty($admin_email)) {
            error_log('CAS Quote Form: No admin email configured');
            return false;
        }
        
        error_log('CAS Quote Form: Sending email to: ' . $admin_email);
        
        $subject = $type === 'booking' 
            ? 'New Booking Received - ' . get_bloginfo('name')
            : 'New Quote Request - ' . get_bloginfo('name');
        
        $message = self::build_email_html($type, $data);
        
        if (empty($message)) {
            error_log('CAS Quote Form: Email message is empty');
            return false;
        }
        
        // Set content type to HTML
        $content_type_filter = function() {
            return 'text/html';
        };
        add_filter('wp_mail_content_type', $content_type_filter);
        
        // Send email using wp_mail (works with Easy SMTP)
        $sent = wp_mail(
            $admin_email,
            $subject,
            $message,
            [
                'Content-Type: text/html; charset=UTF-8',
                'From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>'
            ]
        );
        
        // Remove filter
        remove_filter('wp_mail_content_type', $content_type_filter);
        
        if (!$sent) {
            $last_error = error_get_last();
            error_log('CAS Quote Form: Failed to send admin email notification. Last error: ' . print_r($last_error, true));
        } else {
            error_log('CAS Quote Form: Email sent successfully via wp_mail');
        }
        
        return $sent;
    }
}
