<?php
/**
 * CAS Quote Form - Payments (Stripe Checkout)
 *
 * Secure server-side creation of Checkout Sessions and webhook handling.
 * Reads configuration from the `cas_settings` option (set in the admin Settings page):
 *  - stripe_secret_key
 *  - stripe_publishable_key
 *  - stripe_currency
 *  - stripe_success_url
 *  - stripe_cancel_url
 *  - stripe_webhook_secret
 */

if (!defined('ABSPATH')) { exit; }

// Ensure email helper is loaded
if (!class_exists('CASQuoteEmailHelper')) {
	$email_helper_path = plugin_dir_path(__FILE__) . 'email-helper.php';
	if (file_exists($email_helper_path)) {
		require_once $email_helper_path;
	}
}

class CASQuotePayments {
	public function __construct() {
		add_action('rest_api_init', [$this, 'register_routes']);
	}

	public function register_routes() {
		register_rest_route('cas/v1', '/create-checkout-session', [
			'methods' => 'POST',
			'callback' => [$this, 'create_checkout_session'],
			'permission_callback' => '__return_true',
		]);

		register_rest_route('cas/v1', '/stripe-webhook', [
			'methods' => ['POST', 'GET'],
			'callback' => [$this, 'stripe_webhook'],
			'permission_callback' => '__return_true', // Verified via Stripe signature below
		]);
		
		register_rest_route('cas/v1', '/test-booking-email', [
			'methods' => 'POST',
			'callback' => [$this, 'test_booking_email'],
			'permission_callback' => '__return_true',
		]);
	}

	private function get_setting($key, $default = '') {
		$settings = get_option('cas_settings', []);
		return isset($settings[$key]) && $settings[$key] !== '' ? $settings[$key] : $default;
	}

	// Create a booking record in options for now
	private function create_booking(array $payload) {
		$bookings = get_option('cas_bookings', []);
		$booking_id = wp_generate_uuid4();
		$now = current_time('mysql');
		$bookings[$booking_id] = [
			'id' => $booking_id,
			'status' => 'pending',
			'created_at' => $now,
			'updated_at' => $now,
			'from_port' => $payload['from_port'] ?? '',
			'to_port' => $payload['to_port'] ?? '',
			'from_location' => $payload['from_location'] ?? '',
			'to_location' => $payload['to_location'] ?? '',
			'vehicles' => $payload['vehicles'] ?? [],
			'email' => $payload['email'] ?? '',
			'first_name' => $payload['first_name'] ?? '',
			'last_name' => $payload['last_name'] ?? '',
			'phone' => $payload['phone'] ?? '',
			'pickup_date' => $payload['pickup_date'] ?? '',
			'amount_cents' => (int)($payload['amount_cents'] ?? 0),
			'currency' => $payload['currency'] ?? $this->get_setting('stripe_currency', 'usd'),
			'stripe_session_id' => null,
			'stripe_payment_intent' => null,
			'metadata' => [],
		];
		update_option('cas_bookings', $bookings, false);
		return $bookings[$booking_id];
	}

	private function update_booking($booking_id, array $updates) {
		$bookings = get_option('cas_bookings', []);
		if (!isset($bookings[$booking_id])) { return false; }
		$bookings[$booking_id] = array_merge($bookings[$booking_id], $updates, ['updated_at' => current_time('mysql')]);
		update_option('cas_bookings', $bookings, false);
		return $bookings[$booking_id];
	}

	private function find_price_cents($from_port, $to_port) {
		$pricing = get_option('cas_pricing', []);
		foreach ($pricing as $rule) {
			if (($rule['from_port'] ?? '') === $from_port && ($rule['to_port'] ?? '') === $to_port) {
				$price = (float)$rule['price'];
				return (int) round($price * 100);
			}
		}
		return 0;
	}

	public function create_checkout_session(\WP_REST_Request $request) {
		$stripe_secret = $this->get_setting('stripe_secret_key');
		if (empty($stripe_secret)) {
			return new \WP_REST_Response(['success' => false, 'message' => 'Stripe secret key not configured'], 500);
		}

		$from_port = sanitize_text_field($request->get_param('from_port'));
		$to_port = sanitize_text_field($request->get_param('to_port'));
		$from_location = sanitize_text_field($request->get_param('from_location') ?? $from_port);
		$to_location = sanitize_text_field($request->get_param('to_location') ?? $to_port);
		$email = sanitize_email($request->get_param('email'));
		$first_name = sanitize_text_field($request->get_param('first_name'));
		$last_name = sanitize_text_field($request->get_param('last_name'));
		$phone = sanitize_text_field($request->get_param('phone'));
		$pickup_date = sanitize_text_field($request->get_param('pickup_date') ?? '');
		$vehicles_json = $request->get_param('vehicles');
		$vehicles = [];
		if (!empty($vehicles_json)) {
			$vehicles = json_decode(stripslashes($vehicles_json), true) ?? [];
			// Sanitize vehicle data
			foreach ($vehicles as &$vehicle) {
				$vehicle = [
					'year' => sanitize_text_field($vehicle['year'] ?? ''),
					'make' => sanitize_text_field($vehicle['make'] ?? ''),
					'model' => sanitize_text_field($vehicle['model'] ?? ''),
					'operable' => !empty($vehicle['operable']),
				];
			}
		}

		$amount_cents = $this->find_price_cents($from_port, $to_port);
		if ($amount_cents <= 0) {
			return new \WP_REST_Response(['success' => false, 'message' => 'No pricing available for selected ports'], 400);
		}

		$currency = $this->get_setting('stripe_currency', 'usd');

		$booking = $this->create_booking([
			'from_port' => $from_port,
			'to_port' => $to_port,
			'from_location' => $from_location,
			'to_location' => $to_location,
			'vehicles' => $vehicles,
			'email' => $email,
			'first_name' => $first_name,
			'last_name' => $last_name,
			'phone' => $phone,
			'pickup_date' => $pickup_date,
			'amount_cents' => $amount_cents,
			'currency' => $currency,
		]);

		$success_base = $this->get_setting('stripe_success_url', home_url('/booking/success'));
		$cancel_base  = $this->get_setting('stripe_cancel_url', home_url('/booking/cancel'));
		$success_url = add_query_arg(['bid' => $booking['id']], $success_base);
		$cancel_url  = add_query_arg(['bid' => $booking['id']], $cancel_base);

		// Build Checkout Session via Stripe REST API
		$body = [
			'mode' => 'payment',
			'success_url' => $success_url,
			'cancel_url' => $cancel_url,
			'customer_email' => $email,
			'payment_method_types[]' => 'card',
			'line_items[0][price_data][currency]' => $currency,
			'line_items[0][price_data][unit_amount]' => $amount_cents,
			'line_items[0][price_data][product_data][name]' => sprintf('Auto shipping: %s → %s', $from_port, $to_port),
			'line_items[0][quantity]' => 1,
			'metadata[booking_id]' => $booking['id'],
			'metadata[from_port]' => $from_port,
			'metadata[to_port]' => $to_port,
		];

		$args = [
			'headers' => [
				'Authorization' => 'Bearer ' . $stripe_secret,
			],
			'body' => $body,
			'timeout' => 20,
		];

		$response = wp_remote_post('https://api.stripe.com/v1/checkout/sessions', $args);
		if (is_wp_error($response)) {
			return new \WP_REST_Response(['success' => false, 'message' => $response->get_error_message()], 500);
		}

		$code = wp_remote_retrieve_response_code($response);
		$raw = wp_remote_retrieve_body($response);
		$data = json_decode($raw, true);
		if ($code >= 200 && $code < 300 && !empty($data['url']) && !empty($data['id'])) {
			$this->update_booking($booking['id'], [
				'stripe_session_id' => $data['id'],
				'metadata' => $data['metadata'] ?? [],
			]);
			return new \WP_REST_Response(['success' => true, 'url' => $data['url'], 'booking_id' => $booking['id']], 200);
		}

		return new \WP_REST_Response(['success' => false, 'message' => 'Stripe error', 'details' => $data], 500);
	}

	public function stripe_webhook(\WP_REST_Request $request) {
		$log_file = plugin_dir_path(__FILE__) . 'cas-webhook-debug.log';
		
		// Handle GET requests (for testing)
		if ($_SERVER['REQUEST_METHOD'] === 'GET') {
			return new \WP_REST_Response([
				'success' => true,
				'message' => 'Webhook endpoint is accessible',
				'url' => rest_url('cas/v1/stripe-webhook')
			], 200);
		}
		
		// Log that webhook was called
		@file_put_contents($log_file, 
			date('Y-m-d H:i:s') . " - Webhook called\n" .
			"Request method: " . $_SERVER['REQUEST_METHOD'] . "\n" .
			"Headers: " . print_r(getallheaders(), true) . "\n\n",
			FILE_APPEND
		);
		
		$payload = file_get_contents('php://input');
		$sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';
		$event = null;

		$webhook_secret = $this->get_setting('stripe_webhook_secret');
		
		// Log webhook secret status
		@file_put_contents($log_file,
			date('Y-m-d H:i:s') . " - Webhook secret configured: " . (!empty($webhook_secret) ? 'YES' : 'NO') . "\n",
			FILE_APPEND
		);
		
		// Optional but recommended: verify signature
		if (!empty($webhook_secret)) {
			$timestamp = 0; $signature = '';
			// Parse Stripe-Signature: t=timestamp,v1=signature
			foreach (explode(',', $sig_header) as $part) {
				list($k, $v) = array_pad(explode('=', trim($part), 2), 2, '');
				if ($k === 't') { $timestamp = (int)$v; }
				if ($k === 'v1') { $signature = $v; }
			}
			$expected = hash_hmac('sha256', $timestamp . '.' . $payload, $webhook_secret);
			if (!hash_equals($expected, $signature)) {
				@file_put_contents($log_file,
					date('Y-m-d H:i:s') . " - Invalid webhook signature\n\n",
					FILE_APPEND
				);
				return new \WP_REST_Response(['success' => false, 'message' => 'Invalid webhook signature'], 400);
			}
		}

		$event = json_decode($payload, true);
		$type = $event['type'] ?? '';
		
		// Log event type
		@file_put_contents($log_file,
			date('Y-m-d H:i:s') . " - Event type: $type\n",
			FILE_APPEND
		);

		if ($type === 'checkout.session.completed') {
			@file_put_contents($log_file,
				date('Y-m-d H:i:s') . " - Processing checkout.session.completed event\n",
				FILE_APPEND
			);
			
			$session = $event['data']['object'] ?? [];
			$booking_id = $session['metadata']['booking_id'] ?? '';
			$amount_total = (int)($session['amount_total'] ?? 0);
			$currency = $session['currency'] ?? '';
			$payment_intent = $session['payment_intent'] ?? '';

			@file_put_contents($log_file,
				date('Y-m-d H:i:s') . " - Initial booking_id from metadata: " . ($booking_id ?: 'EMPTY') . "\n" .
				"Session metadata: " . print_r($session['metadata'] ?? [], true) . "\n",
				FILE_APPEND
			);

			if (empty($booking_id)) {
				// Try alternative metadata keys
				$booking_id = $session['metadata']['booking_id'] ?? $session['client_reference_id'] ?? '';
				@file_put_contents($log_file,
					date('Y-m-d H:i:s') . " - Tried alternative keys, booking_id: " . ($booking_id ?: 'EMPTY') . "\n",
					FILE_APPEND
				);
			}

			if ($booking_id) {
				// Get booking data BEFORE updating (to preserve original data)
				$bookings = get_option('cas_bookings', []);
				$booking = $bookings[$booking_id] ?? null;
				
				if (!$booking) {
					// Log for debugging
					@file_put_contents($log_file, 
						date('Y-m-d H:i:s') . " - Booking not found: $booking_id\n" . 
						"Available bookings: " . implode(', ', array_keys($bookings)) . "\n" .
						"Session metadata: " . print_r($session['metadata'] ?? [], true) . "\n\n",
						FILE_APPEND
					);
					return new \WP_REST_Response(['received' => true, 'error' => 'Booking not found', 'booking_id' => $booking_id], 200);
				}

				// Update booking status
				$this->update_booking($booking_id, [
					'status' => 'paid',
					'stripe_payment_intent' => $payment_intent,
					'amount_cents' => $amount_total ?: ($booking['amount_cents'] ?? 0),
					'currency' => $currency ?: ($booking['currency'] ?? $this->get_setting('stripe_currency', 'usd')),
				]);
				
				// Prepare email data
				$email_data = [
					'booking_id' => $booking_id,
					'from_location' => $booking['from_location'] ?? $booking['from_port'] ?? '',
					'to_location' => $booking['to_location'] ?? $booking['to_port'] ?? '',
					'vehicles' => $booking['vehicles'] ?? [],
					'first_name' => $booking['first_name'] ?? '',
					'last_name' => $booking['last_name'] ?? '',
					'email' => $booking['email'] ?? '',
					'phone' => $booking['phone'] ?? '',
					'pickup_date' => $booking['pickup_date'] ?? '',
					'amount_cents' => $amount_total ?: ($booking['amount_cents'] ?? 0),
					'currency' => $currency ?: ($booking['currency'] ?? 'usd'),
				];
				
				// Ensure email helper is loaded
				if (!class_exists('CASQuoteEmailHelper')) {
					$email_helper_path = plugin_dir_path(__FILE__) . 'email-helper.php';
					if (file_exists($email_helper_path)) {
						require_once $email_helper_path;
					}
				}
				
				// Send admin email notification
				if (class_exists('CASQuoteEmailHelper')) {
					$email_sent = CASQuoteEmailHelper::send_admin_notification('booking', $email_data);
					
					// Log email sending attempt
					@file_put_contents($log_file,
						date('Y-m-d H:i:s') . " - Email sent for booking $booking_id: " . ($email_sent ? 'YES' : 'NO') . "\n" .
						"Email data: " . print_r($email_data, true) . "\n\n",
						FILE_APPEND
					);
				} else {
					@file_put_contents($log_file,
						date('Y-m-d H:i:s') . " - Email helper class not found for booking $booking_id\n\n",
						FILE_APPEND
					);
				}
			} else {
				@file_put_contents($log_file,
					date('Y-m-d H:i:s') . " - No booking_id in webhook\n" .
					"Session data: " . print_r($session, true) . "\n\n",
					FILE_APPEND
				);
			}
		}

		return new \WP_REST_Response(['received' => true], 200);
	}

	private function get_booking_amount($booking_id) {
		$bookings = get_option('cas_bookings', []);
		return isset($bookings[$booking_id]) ? (int)$bookings[$booking_id]['amount_cents'] : 0;
	}
	
	/**
	 * Test endpoint to verify email sending works
	 */
	public function test_booking_email($request) {
		if (!class_exists('CASQuoteEmailHelper')) {
			return new \WP_REST_Response(['success' => false, 'message' => 'Email helper not loaded'], 500);
		}
		
		$test_data = [
			'booking_id' => 'test-' . time(),
			'from_location' => 'Los Angeles, CA',
			'to_location' => 'Miami, FL',
			'vehicles' => [
				['year' => '2024', 'make' => 'Toyota', 'model' => 'Camry', 'operable' => true]
			],
			'first_name' => 'Test',
			'last_name' => 'User',
			'email' => 'test@example.com',
			'phone' => '555-1234',
			'pickup_date' => '2024-12-25',
			'amount_cents' => 150000,
			'currency' => 'usd',
		];
		
		$sent = CASQuoteEmailHelper::send_admin_notification('booking', $test_data);
		
		return new \WP_REST_Response([
			'success' => $sent,
			'message' => $sent ? 'Test email sent' : 'Failed to send test email'
		], $sent ? 200 : 500);
	}
}

new CASQuotePayments();
