import alpine from 'alpinejs';

// Import wizard functionality BEFORE starting Alpine
import './quote/wizard';

// Register Select2 component
alpine.data('select2', () => ({
  open: false,
  search: '',
  selected: null,
  options: [],
  
  init() {
    this.options = this.$el.dataset.options ? JSON.parse(this.$el.dataset.options) : [];
  },
  
  get filteredOptions() {
    if (!this.search || this.search.trim() === '') {
      return this.options;
    }
    return this.options.filter(option => 
      option.label.toLowerCase().includes(this.search.toLowerCase())
    );
  },
  
  select(option) {
    this.selected = option;
    this.open = false;
    this.search = '';
    this.$dispatch('select', option);
  },
  
  toggle() {
    this.open = !this.open;
    if (this.open) {
      this.$nextTick(() => {
        this.$refs.search.focus();
      });
    }
  },
  
  close() {
    this.open = false;
    this.search = '';
  },
  
  clear() {
    this.selected = null;
    this.search = '';
    this.open = false;
  },
}));

// Register Google Places Autocomplete component
alpine.data('placesAutocomplete', (fieldName) => ({
  suggestions: [],
  showSuggestions: false,
  debounceTimer: null,
  
  init() {
    console.log('Places Autocomplete initialized for field:', fieldName);
  },
  
  async handleInput(event) {
    const input = event.target.value;
    
    // Clear previous timer
    if (this.debounceTimer) {
      clearTimeout(this.debounceTimer);
    }
    
    // Debounce API calls
    this.debounceTimer = setTimeout(async () => {
      if (input.length >= 2) {
        await this.fetchSuggestions(input);
      } else {
        this.suggestions = [];
      }
    }, 300);
  },
  
  async fetchSuggestions(input) {
    try {
      const response = await fetch(`${window.location.origin}/wp-json/cas/v1/places-autocomplete?input=${encodeURIComponent(input)}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
        },
      });
      
      const data = await response.json();
      
      if (data.success && data.data.predictions) {
        const predictions = data.data.predictions.slice(0, 5); // Limit to 5 suggestions
        
        // Fetch details for each suggestion to get complete addresses with ZIP codes
        const suggestionsWithDetails = await Promise.all(
          predictions.map(async (prediction) => {
            try {
              const detailsResponse = await fetch(`${window.location.origin}/wp-json/cas/v1/places-details?place_id=${encodeURIComponent(prediction.place_id)}`, {
                method: 'GET',
                headers: {
                  'Accept': 'application/json',
                },
              });
              
              const detailsData = await detailsResponse.json();
              
              if (detailsData.success && detailsData.data.result) {
                const place = detailsData.data.result;
                const fullAddress = place.formatted_address || prediction.description;
                const addressWithoutCountry = fullAddress.replace(/, USA$/, '').replace(/, United States$/, '');
                
                return {
                  ...prediction,
                  description: addressWithoutCountry,
                  full_address: addressWithoutCountry,
                };
              } else {
                // Fallback to original description without country
                const addressWithoutCountry = prediction.description.replace(/, USA$/, '').replace(/, United States$/, '');
                return {
                  ...prediction,
                  description: addressWithoutCountry,
                  full_address: addressWithoutCountry,
                };
              }
            } catch (error) {
              console.error('Error fetching details for suggestion:', error);
              // Fallback to original description without country
              const addressWithoutCountry = prediction.description.replace(/, USA$/, '').replace(/, United States$/, '');
              return {
                ...prediction,
                description: addressWithoutCountry,
                full_address: addressWithoutCountry,
              };
            }
          })
        );
        
        this.suggestions = suggestionsWithDetails;
        console.log('Suggestions with details received:', this.suggestions);
      } else {
        this.suggestions = [];
      }
    } catch (error) {
      console.error('Error fetching places suggestions:', error);
      this.suggestions = [];
    }
  },
  
  async selectSuggestion(suggestion) {
    // Use the pre-fetched complete address
    console.log('Selected suggestion:', suggestion.full_address);
    
    // Update the model with the complete address
    this.$dispatch('place-selected', { field: fieldName, value: suggestion.full_address });
    this.showSuggestions = false;
    this.suggestions = [];
  },
  
  formatSuggestionDisplay(suggestion) {
    // Display suggestion description without country since we're US-only
    return suggestion.description.replace(/, USA$/, '').replace(/, United States$/, '');
  },
}));

// Register the wizard component directly with Alpine before starting
alpine.data('casqWizard', () => {
  const STORAGE = 'casq_wizard_v1'
  
  return {
    step: 1,
    maxStepReached: 1,
        model: {
          from_city_zip: '',
          to_city_zip: '',
          dest_port: '',
          origin_mode: 'city',
          dest_mode: 'city',
          vehicles: [
            { id: 'vehicle-1', year: '', make: '', model: '', operable: true },
          ],
          first_name: '',
          last_name: '',
          email: '',
          phone: '',
          pickup_date: '',
          consent_sms: false,
          quote_cents: 0,
          currency: 'usd',
          token: '',
          entry_id: null,
        },
        // Vehicle dropdown data
        vehicleYears: [],
        yearOptions: [],
        // Per-vehicle options storage: { 'vehicle-1': { makes: [], models: [] } }
        vehicleOptions: {},
        loadingYears: false,
            portOptions: [],
            fromPortOptions: [],
            destPortOptions: [],
            pricing: [],
    errors: {},
    busy: false,

        init() {
          console.log('Wizard init called, current step:', this.step)
          const saved = localStorage.getItem(STORAGE)
          if (saved) {
            try {
              const s = JSON.parse(saved)
              this.step = s.step || 1
              this.maxStepReached = s.maxStepReached || 1
              const loadedModel = s.model || {}
              
              // Migrate old single vehicle format to vehicles array
              if (!loadedModel.vehicles && (loadedModel.vehicle_year || loadedModel.vehicle_make || loadedModel.vehicle_model)) {
                loadedModel.vehicles = [{
                  id: 'vehicle-1',
                  year: loadedModel.vehicle_year || '',
                  make: loadedModel.vehicle_make || '',
                  model: loadedModel.vehicle_model || '',
                  operable: loadedModel.operable !== undefined ? loadedModel.operable : true,
                }]
                // Clean up old fields
                delete loadedModel.vehicle_year
                delete loadedModel.vehicle_make
                delete loadedModel.vehicle_model
                delete loadedModel.operable
              } else if (!loadedModel.vehicles) {
                loadedModel.vehicles = [{ id: 'vehicle-1', year: '', make: '', model: '', operable: true }]
              }
              
              // Ensure all vehicles have IDs
              loadedModel.vehicles = loadedModel.vehicles.map((v, idx) => ({
                ...v,
                id: v.id || `vehicle-${idx + 1}`,
              }))
              
              Object.assign(this.model, loadedModel)
              console.log('Loaded from localStorage:', s)
            } catch (e) {
              console.warn('Failed to parse saved wizard data:', e)
            }
          }

          // Initialize vehicle options storage and load options for vehicles with data
          this.model.vehicles.forEach(vehicle => {
            if (!this.vehicleOptions[vehicle.id]) {
              this.vehicleOptions[vehicle.id] = { makes: [], models: [] }
            }
            // If vehicle has year, load makes
            if (vehicle.year) {
              this.loadVehicleMakesForVehicle(vehicle.id, vehicle.year)
            }
            // If vehicle has year and make, load models
            if (vehicle.year && vehicle.make) {
              this.loadVehicleModelsForVehicle(vehicle.id, vehicle.year, vehicle.make)
            }
          })

          // Load vehicle years on init
          this.loadVehicleYears()

          // Initialize filtered port options
          this.updatePortOptionFilters()

            // Load ports from backend
            this.loadPorts()
            
            // Load pricing from backend
            this.loadPricing()
            
            // Listen for place selection events
          this.$el.addEventListener('place-selected', (event) => {
            const { field, value } = event.detail;
            this.model[field] = value;
            console.log('Place selected:', field, value);
          });
        },

        get quotePrice() {
          // Check if both origin and destination are ports
          if (this.model.origin_mode === 'port' && this.model.dest_mode === 'port') {
            const fromPort = this.model.from_city_zip
            const toPort = this.model.dest_port
            const vehicleCount = this.model.vehicles.length
            
            console.log('Looking for pricing:', { fromPort, toPort, vehicleCount })
            console.log('Available pricing:', this.pricing)
            
            // Find matching pricing rule
            const pricingRule = this.pricing.find(rule => 
              rule.from_port === fromPort && rule.to_port === toPort
            )
            
            if (pricingRule) {
              console.log('Found pricing rule:', pricingRule)
              const perVehiclePrice = pricingRule.price
              const totalPrice = perVehiclePrice * vehicleCount
              return {
                found: true,
                price: totalPrice,
                perVehicle: perVehiclePrice,
                vehicleCount: vehicleCount,
                formatted: `$${totalPrice.toLocaleString('en-US', { maximumFractionDigits: 0 })}`,
                message: 'The quoted price includes all applicable taxes and fees.',
              }
            } else {
              console.log('No pricing rule found')
              return {
                found: false,
                message: 'Quote will be emailed to you',
              }
            }
          }
          
          // If not port-to-port, show default message
          return {
            found: false,
            message: 'Quote will be emailed to you',
          }
        },

    save() {
      localStorage.setItem(STORAGE, JSON.stringify({
        step: this.step,
        maxStepReached: this.maxStepReached,
        model: this.model,
      }))
    },

    back() {
      this.step = Math.max(1, this.step - 1)
      this.save()
    },

    async next() {
      console.log('Next clicked, current step:', this.step)
      console.log('Model values:', this.model)
      if (!this.validate()) {
        console.log('Validation failed, errors:', this.errors)
        return
      }
      
      const nextStep = Math.min(5, this.step + 1)
      
      // If moving to step 4 and no price is available, submit quote request
      if (nextStep === 4 && !this.quotePrice.found) {
        await this.submitQuoteRequest()
      }
      
      this.step = nextStep
      this.maxStepReached = Math.max(this.maxStepReached, this.step)
      console.log('Moving to step:', this.step)
      this.save()
    },
    
    async submitQuoteRequest() {
      this.busy = true
      try {
        const params = new URLSearchParams({
          origin_mode: this.model.origin_mode || 'city',
          dest_mode: this.model.dest_mode || 'city',
          from_city_zip: this.model.from_city_zip || '',
          to_city_zip: this.model.to_city_zip || '',
          from_port: (this.model.origin_mode === 'port' ? this.model.from_city_zip : '') || '',
          to_port: this.model.dest_port || '',
          first_name: this.model.first_name || '',
          last_name: this.model.last_name || '',
          email: this.model.email || '',
          phone: this.model.phone || '',
          pickup_date: this.model.pickup_date || '',
          vehicles: JSON.stringify(this.model.vehicles || []),
        })
        
        const response = await fetch(`${window.location.origin}/wp-json/cas/v1/submit-quote`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
          body: params,
        })
        
        const data = await response.json()
        if (data.success) {
          console.log('Quote request submitted successfully')
        } else {
          console.error('Failed to submit quote request:', data.message)
        }
      } catch (error) {
        console.error('Error submitting quote request:', error)
      } finally {
        this.busy = false
      }
    },

    validate() {
      this.errors = {}
      return true // Always allow navigation for now
    },

    async fetchQuote() {
      console.log('Fetching quote...', this.model)
      this.busy = true
      
      try {
        // Simulate API call - replace with actual quote API
        await new Promise(resolve => setTimeout(resolve, 2000))
        
        // Mock quote calculation - replace with real API call
        const mockQuote = Math.floor(Math.random() * 1000) + 500 // Random between $500-$1500
        this.model.quote_cents = mockQuote * 100 // Convert to cents
        
        console.log('Quote generated:', this.model.quote_cents)
        
        // Move to next step after quote is generated
        this.next()
        
      } catch (error) {
        console.error('Error fetching quote:', error)
        // Handle error - maybe show error message
      } finally {
        this.busy = false
      }
    },

    async mountStripe() {
      console.log('Mounting Stripe...')
    },

    async pay() {
      console.log('Processing payment...')
    },

    // Vehicle API methods using WordPress REST API
    async loadVehicleYears() {
      this.loadingYears = true
      try {
        console.log('Attempting to load vehicle years...')
        const response = await fetch(`${window.location.origin}/wp-json/cas/v1/vehicle-years`, {
          method: 'GET',
          headers: {
            'Accept': 'application/json',
          },
        })
        
        console.log('Response status:', response.status)
        
        if (!response.ok) {
          const errorText = await response.text()
          console.error('API Error Response:', errorText)
          throw new Error(`HTTP error! status: ${response.status}, body: ${errorText}`)
        }
        
        const data = await response.json()
        console.log('Response data:', data)
        
        if (data.success) {
          this.vehicleYears = data.data.years || []
          // Format years for Select2 component
          this.yearOptions = this.vehicleYears.map(year => ({
            value: year,
            label: year.toString(),
          }))
          console.log('Loaded vehicle years:', this.vehicleYears)
          console.log('Formatted yearOptions:', this.yearOptions)
          // Clear any previous errors
          delete this.errors.vehicle_years
        } else {
          throw new Error(data.message || 'Failed to load vehicle years')
        }
      } catch (error) {
        console.error('Error loading vehicle years:', error)
        this.vehicleYears = []
        this.errors.vehicle_years = 'Failed to load vehicle years. Please try again.'
      } finally {
        this.loadingYears = false
      }
    },

    async loadPorts() {
      try {
        console.log('Loading ports from backend...')
        const response = await fetch(`${window.location.origin}/wp-json/cas/v1/ports`, {
          method: 'GET',
          headers: {
            'Accept': 'application/json',
          },
        })
        
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`)
        }
        
        const data = await response.json()
        console.log('Ports response data:', data)
        
        if (data.success) {
          const ports = data.data || []
          // Format ports for Select2 component: "Port Name, State"
          this.portOptions = ports.map(port => ({
            value: `${port.name}, ${port.state}`,
            label: `${port.name}, ${port.state}`,
          }))
          this.updatePortOptionFilters()
          console.log('Loaded ports:', this.portOptions)
        } else {
          console.error('Failed to load ports:', data.message)
        }
      } catch (error) {
        console.error('Error loading ports:', error)
      }
    },

    async loadPricing() {
      try {
        console.log('Loading pricing from backend...')
        const response = await fetch(`${window.location.origin}/wp-json/cas/v1/pricing`, {
          method: 'GET',
          headers: {
            'Accept': 'application/json',
          },
        })
        
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`)
        }
        
        const data = await response.json()
        console.log('Pricing response data:', data)
        
        if (data.success) {
          this.pricing = data.data || []
          console.log('Loaded pricing:', this.pricing)
        } else {
          console.error('Failed to load pricing:', data.message)
        }
      } catch (error) {
        console.error('Error loading pricing:', error)
      }
    },

    async loadVehicleMakesForVehicle(vehicleId, year) {
      if (!year || !vehicleId) return
      
      // Initialize vehicle options if not exists
      if (!this.vehicleOptions[vehicleId]) {
        this.vehicleOptions[vehicleId] = { makes: [], models: [] }
      }
      
      try {
        const response = await fetch(`${window.location.origin}/wp-json/cas/v1/vehicle-makes?year=${year}`, {
          method: 'GET',
          headers: {
            'Accept': 'application/json',
          },
        })
        
        const data = await response.json()
        
        if (data.success) {
          const makes = data.data.makes || []
          // Store makes for this vehicle
          this.vehicleOptions[vehicleId].makes = makes.map(make => ({
            value: make,
            label: make,
          }))
          console.log(`Loaded makes for vehicle ${vehicleId}, year ${year}:`, makes)
        } else {
          throw new Error(data.message || 'Failed to load vehicle makes')
        }
      } catch (error) {
        console.error(`Error loading makes for vehicle ${vehicleId}:`, error)
        this.vehicleOptions[vehicleId].makes = []
      }
    },

    async loadVehicleModelsForVehicle(vehicleId, year, make) {
      if (!year || !make || !vehicleId) return
      
      // Initialize vehicle options if not exists
      if (!this.vehicleOptions[vehicleId]) {
        this.vehicleOptions[vehicleId] = { makes: [], models: [] }
      }
      
      try {
        const response = await fetch(`${window.location.origin}/wp-json/cas/v1/vehicle-models?year=${year}&make=${make}`, {
          method: 'GET',
          headers: {
            'Accept': 'application/json',
          },
        })
        
        const data = await response.json()
        
        if (data.success) {
          const models = data.data.models || []
          // Store models for this vehicle
          this.vehicleOptions[vehicleId].models = models.map(model => ({
            value: model,
            label: model,
          }))
          console.log(`Loaded models for vehicle ${vehicleId}, ${year} ${make}:`, models)
        } else {
          throw new Error(data.message || 'Failed to load vehicle models')
        }
      } catch (error) {
        console.error(`Error loading models for vehicle ${vehicleId}:`, error)
        this.vehicleOptions[vehicleId].models = []
      }
    },

    // Handle year selection from Select2 for a specific vehicle
    handleYearSelect(event, vehicleId) {
      const selectedYear = event.detail.value
      const vehicle = this.model.vehicles.find(v => v.id === vehicleId)
      if (!vehicle) return
      
      vehicle.year = selectedYear
      
      // Reset Make and Model when year changes
      vehicle.make = ''
      vehicle.model = ''
      if (this.vehicleOptions[vehicleId]) {
        this.vehicleOptions[vehicleId].models = []
      }
      
      // Load makes for this vehicle
      this.loadVehicleMakesForVehicle(vehicleId, selectedYear)
      this.save()
    },

    // Handle make selection from Select2 for a specific vehicle
    handleMakeSelect(event, vehicleId) {
      const selectedMake = event.detail.value
      const vehicle = this.model.vehicles.find(v => v.id === vehicleId)
      if (!vehicle) return
      
      vehicle.make = selectedMake
      
      // Reset Model when make changes
      vehicle.model = ''
      if (this.vehicleOptions[vehicleId]) {
        this.vehicleOptions[vehicleId].models = []
      }
      
      // Load models for this vehicle
      this.loadVehicleModelsForVehicle(vehicleId, vehicle.year, selectedMake)
      this.save()
    },

    // Handle model selection from Select2 for a specific vehicle
    handleModelSelect(event, vehicleId) {
      const selectedModel = event.detail.value
      const vehicle = this.model.vehicles.find(v => v.id === vehicleId)
      if (!vehicle) return
      
      vehicle.model = selectedModel
      this.save()
    },

    // Add a new vehicle (max 5)
    addVehicle() {
      if (this.model.vehicles.length >= 5) return
      
      const newId = `vehicle-${Date.now()}`
      this.model.vehicles.push({
        id: newId,
        year: '',
        make: '',
        model: '',
        operable: true,
      })
      
      // Initialize options for new vehicle
      this.vehicleOptions[newId] = { makes: [], models: [] }
      this.save()
    },

    // Remove a vehicle (min 1)
    removeVehicle(vehicleId) {
      if (this.model.vehicles.length <= 1) return
      
      this.model.vehicles = this.model.vehicles.filter(v => v.id !== vehicleId)
      delete this.vehicleOptions[vehicleId]
      this.save()
    },

        // Handle port selection from Select2
        handlePortSelect(event, field) {
          const selectedPort = event.detail.value
          if (field === 'from') {
            this.model.from_city_zip = selectedPort
            if (this.model.dest_port === selectedPort) {
              this.model.dest_port = ''
            }
          } else if (field === 'dest') {
            this.model.dest_port = selectedPort
            if (this.model.from_city_zip === selectedPort) {
              this.model.from_city_zip = ''
            }
          }
          this.updatePortOptionFilters()
        },

        updatePortOptionFilters() {
          if (!Array.isArray(this.portOptions)) {
            this.fromPortOptions = []
            this.destPortOptions = []
            return
          }

          if (this.model.from_city_zip && this.model.dest_port === this.model.from_city_zip) {
            this.model.dest_port = ''
          }

          this.fromPortOptions = this.portOptions.filter(option => option.value !== this.model.dest_port)
          this.destPortOptions = this.portOptions.filter(option => option.value !== this.model.from_city_zip)
        },

      async startCheckout() {
        if (this.busy) return
        this.busy = true
        try {
          const fromPort = this.model.origin_mode === 'port' ? this.model.from_city_zip : ''
          const toPort = this.model.dest_mode === 'port' ? this.model.dest_port : ''
          const fromLocation = this.model.origin_mode === 'port' ? fromPort : this.model.from_city_zip
          const toLocation = this.model.dest_mode === 'port' ? toPort : this.model.to_city_zip
          
          const params = new URLSearchParams({
            from_port: fromPort || '',
            to_port: toPort || '',
            from_location: fromLocation || '',
            to_location: toLocation || '',
            email: this.model.email || '',
            first_name: this.model.first_name || '',
            last_name: this.model.last_name || '',
            phone: this.model.phone || '',
            pickup_date: this.model.pickup_date || '',
            vehicles: JSON.stringify(this.model.vehicles || []),
          })
          const res = await fetch(`${window.location.origin}/wp-json/cas/v1/create-checkout-session`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: params,
          })
          const data = await res.json()
          if (!res.ok || !data.success || !data.url) {
            console.error('Checkout error', data)
            alert(data.message || 'Unable to start checkout.')
            this.busy = false
            return
          }
          window.location.href = data.url
        } catch (e) {
          console.error('Checkout exception', e)
          alert('Unable to start checkout.')
        } finally {
          this.busy = false
        }
      },

      cancelPayment() {
        // Clear localStorage
        localStorage.removeItem(STORAGE)
        
        // Reload the page to start fresh
        window.location.reload()
      },
  }
})

// Register Hero Form component
alpine.data('heroForm', () => {
  return {
    busy: false,
    errors: {},
    portOptions: [],
    fromPortOptions: [],
    destPortOptions: [],
    model: {
      origin_mode: 'city', // Default to City/ZIP as per user request
      dest_mode: 'city',
      from_city_zip: '',
      to_city_zip: '',
      dest_port: '',
    },

    async init() {
      console.log('Hero form initialized');
      await this.loadPorts();
      this.updatePortOptionFilters();
    },

    async loadPorts() {
      try {
        const response = await fetch(`${window.location.origin}/wp-json/cas/v1/ports`, {
          method: 'GET',
          headers: {
            'Accept': 'application/json',
          },
        });
        
        const data = await response.json();
        
        if (data.success && data.data) {
          this.portOptions = data.data.map(port => ({
            value: `${port.name}, ${port.state}`,
            label: `${port.name}, ${port.state}`,
          }));
          this.updatePortOptionFilters();
          console.log('Loaded ports for hero form:', this.portOptions);
        }
      } catch (error) {
        console.error('Error loading ports for hero form:', error);
      }
    },

    handlePortSelect(event, type) {
      const selectedPort = event.detail;
      if (type === 'from') {
        this.model.from_city_zip = selectedPort.value;
        if (this.model.dest_port === selectedPort.value) {
          this.model.dest_port = '';
        }
      } else if (type === 'dest') {
        this.model.dest_port = selectedPort.value;
        if (this.model.from_city_zip === selectedPort.value) {
          this.model.from_city_zip = '';
        }
      }
      this.updatePortOptionFilters();
    },

    updatePortOptionFilters() {
      if (!Array.isArray(this.portOptions)) {
        this.fromPortOptions = [];
        this.destPortOptions = [];
        return;
      }

      if (this.model.from_city_zip && this.model.dest_port === this.model.from_city_zip) {
        this.model.dest_port = '';
      }

      this.fromPortOptions = this.portOptions.filter(option => option.value !== this.model.dest_port);
      this.destPortOptions = this.portOptions.filter(option => option.value !== this.model.from_city_zip);
    },

    validate() {
      this.errors = {};
      
      // Validate origin
      if (this.model.origin_mode === 'city' && !this.model.from_city_zip.trim()) {
        this.errors.from_city_zip = 'Please enter a city or ZIP code';
      } else if (this.model.origin_mode === 'port' && !this.model.from_city_zip.trim()) {
        this.errors.from_city_zip = 'Please select a port';
      }
      
      // Validate destination
      if (this.model.dest_mode === 'city' && !this.model.to_city_zip.trim()) {
        this.errors.to_city_zip = 'Please enter a city or ZIP code';
      } else if (this.model.dest_mode === 'port' && !this.model.dest_port.trim()) {
        this.errors.dest_port = 'Please select a port';
      }
      
      return Object.keys(this.errors).length === 0;
    },

    async proceedToWizard() {
      if (!this.validate()) {
        return;
      }

      this.busy = true;
      
      try {
        // Store the form data in localStorage for the wizard to pick up
        const wizardData = {
          step: 2, // Start at step 2 (Vehicle Details)
          maxStepReached: 2,
          model: {
            origin_mode: this.model.origin_mode,
            dest_mode: this.model.dest_mode,
            from_city_zip: this.model.from_city_zip,
            to_city_zip: this.model.to_city_zip,
            dest_port: this.model.dest_port,
          },
        };
        
        localStorage.setItem('casq_wizard_v1', JSON.stringify(wizardData));
        
        // Redirect to the wizard page
        window.location.href = '/instant-shipping-quote/';
      } catch (error) {
        console.error('Error proceeding to wizard:', error);
        alert('Unable to proceed. Please try again.');
      } finally {
        this.busy = false;
      }
    },
  }
});

Object.assign(window, {Alpine: alpine}).Alpine.start();

import.meta.webpackHot?.accept(console.error);
