import {
  useBlockProps,
  InspectorControls,
} from "@wordpress/block-editor";
import { PanelBody, TextControl, TextareaControl } from "@wordpress/components";
import { __ } from "@wordpress/i18n";

/* Block name */
export const name = `radicle/cta`;

/* Block title */
export const title = __(`Call to Action`, `radicle`);

/* Block category */
export const category = `design`;

/* Block attributes */
export const attributes = {
  heading: {
    type: `string`,
    default: __(`CTA Heading`, `radicle`),
  },
  description: {
    type: `string`,
    default: __(`Add your call to action description here.`, `radicle`),
  },
  primaryButtonText: {
    type: `string`,
    default: __(`Primary Button`, `radicle`),
  },
  primaryButtonUrl: {
    type: `string`,
    default: `#`,
  },
  image: {
    type: `object`,
    default: null,
  },
};

// Import MediaUpload and MediaUploadCheck
import { MediaUpload, MediaUploadCheck } from "@wordpress/block-editor";

/* Block edit */
export const edit = ({ attributes, setAttributes }) => {
  const props = useBlockProps({
    className: `border p-6`,
  });
  
  const { 
    heading, 
    description, 
    primaryButtonText, 
    primaryButtonUrl,
    image,
  } = attributes;

  return (
    <>
      <InspectorControls>
        <PanelBody title={__(`CTA Settings`, `radicle`)}>
          <MediaUploadCheck>
            <MediaUpload
              onSelect={(media) =>
                setAttributes({ image: { url: media.url, alt: media.alt, title: media.title } })
              }
              allowedTypes={['image']}
              value={image?.id}
              render={({ open }) => (
                <div style={{ marginBottom: '1rem' }}>
                  {image?.url ? (
                    <>
                      <img
                        src={image.url}
                        alt={image.alt || ''}
                        title={image.title || ''}
                        style={{ maxWidth: '100%', height: 'auto', width: '100%', marginBottom: '0.5rem' }}
                        onClick={open}
                      />
                      <button
                        type="button"
                        className="button button-secondary"
                        onClick={() => setAttributes({ image: null })}
                      >
                        {__('Remove Image', 'radicle')}
                      </button>
                    </>
                  ) : (
                    <button onClick={open} className="button button-primary">
                      {__('Add Image', 'radicle')}
                    </button>
                  )}
                </div>
              )}
            />
          </MediaUploadCheck>
          <TextControl
            label={__(`Heading`, `radicle`)}
            value={heading}
            onChange={(heading) => setAttributes({ heading })}
          />
          <TextareaControl
            label={__(`Description`, `radicle`)}
            value={description}
            onChange={(description) => setAttributes({ description })}
          />
        </PanelBody>
        <PanelBody title={__(`Button Settings`, `radicle`)}>
          <TextControl
            label={__(`Primary Button Text`, `radicle`)}
            value={primaryButtonText}
            onChange={(primaryButtonText) => setAttributes({ primaryButtonText })}
          />
          <TextControl
            label={__(`Primary Button URL`, `radicle`)}
            value={primaryButtonUrl}
            onChange={(primaryButtonUrl) => setAttributes({ primaryButtonUrl })}
          />
        </PanelBody>
      </InspectorControls>

      {image?.url && (
        <div>
          <img
            src={image.url}
            alt={image.alt || ''}
            title={image.title || ''}
            style={{ maxWidth: '100%', height: 'auto', width: '100%' }}
            loading="lazy"
          />
        </div>
      )}
      <div {...props}>
        <h2 className="mt-0">{heading}</h2>
        <p>{description}</p>
        <a 
          href={primaryButtonUrl}
          className="no-underline"
        >
          {primaryButtonText}
        </a>
      </div>
    </>
  );
};

/* Block save */
export const save = ({ attributes }) => {
  const { 
    heading, 
    description,
    primaryButtonText,
    primaryButtonUrl,
    image,
  } = attributes;
  const blockProps = useBlockProps.save();
  
  return (
    <div {...blockProps}>
      {image?.url && (
        <img src={image.url} alt={image.alt || ''} title={image.title || ''} />
      )}
      <h2>{heading}</h2>
      <p>{description}</p>
      <div className="buttons">
        <a href={primaryButtonUrl}>{primaryButtonText}</a>
      </div>
    </div>
  );
}; 